# -*- coding: utf-8 -*-
# Copyright 2022 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function


__metaclass__ = type

"""
The aeos7 hostname fact class
It is in this file the configuration is collected from the device
for a given resource, parsed, and the facts tree is populated
based on the configuration.
"""

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common import utils

from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.rm_templates.hostname import (
    HostnameTemplate,
)


class HostnameFacts(object):
    """The aeos7 hostname facts class"""

    def __init__(self, module, subspec="config", options="options"):
        self._module = module

    def get_hostname_data(self, connection):
        return connection.get("show running-config | include hostname")

    def populate_facts(self, connection, ansible_facts, data=None):
        """Populate the facts for Hostname network resource

        :param connection: the device connection
        :param ansible_facts: Facts dictionary
        :param data: previously collected conf

        :rtype: dictionary
        :returns: facts
        """

        if not data:
            data = self.get_hostname_data(connection)

        # parse native config using the Hostname template
        hostname_parser = HostnameTemplate(
            lines=data.splitlines(), module=self._module)
        objs = hostname_parser.parse()

        params = utils.remove_empties({"config": objs})

        facts = {"hostname": params.get("config", {})}
        ansible_facts["ansible_network_resources"].pop("hostname", None)
        ansible_facts["ansible_network_resources"].update(facts)

        return ansible_facts
