# -*- coding: utf-8 -*-
# Copyright 2022 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function


__metaclass__ = type

"""
The aeos7 interfaces fact class
It is in this file the configuration is collected from the device
for a given resource, parsed, and the facts tree is populated
based on the configuration.
"""


import re

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common import utils

from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.rm_templates.interfaces import (
    InterfacesTemplate,
)


class InterfacesFacts(object):
    """The aeos7 interfaces fact class"""

    def __init__(self, module, subspec="config", options="options"):
        self._module = module

    def get_interfaces_data(self, connection):
        return connection.get("show running-config")

    def populate_facts(self, connection, ansible_facts, data=None):
        """Populate the facts for interfaces

        :param connection: the device connection
        :param ansible_facts: Facts dictionary
        :param data: previously collected conf

        :rtype: dictionary
        :returns: facts
        """
        facts = {"interfaces": []}

        if not data:
            data = self.get_interfaces_data(connection)

        for interface in self.iter_interfaces(data):
            interfaces_parser = InterfacesTemplate(
                lines=interface.splitlines(), module=self._module)
            objs = interfaces_parser.parse()

            if objs.get("enabled", None) is None:
                objs["enabled"] = True

            facts["interfaces"].append(utils.remove_empties(objs))

        ansible_facts["ansible_network_resources"].pop("interfaces", None)
        ansible_facts["ansible_network_resources"].update(facts)

        return ansible_facts

    def iter_interfaces(self, data):
        interfaces = re.compile(
            r"^interface port.+?(?=^!)", re.DOTALL | re.MULTILINE)
        return (i.group(0) for i in interfaces.finditer(data))
