# -*- coding: utf-8 -*-
# Copyright 2022 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function


__metaclass__ = type

"""
The aeos7 l2_interfaces fact class
It is in this file the configuration is collected from the device
for a given resource, parsed, and the facts tree is populated
based on the configuration.
"""

import re

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common import utils

from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.rm_templates.l2_interfaces import (
    L2_InterfacesTemplate,
)


class L2_InterfacesFacts(object):
    """The aeos7 l2 interfaces fact class"""

    def __init__(self, module, subspec="config", options="options"):
        self._module = module

    def get_l2_interfaces_data(self, connection):
        return connection.get("show running-config")

    def populate_facts(self, connection, ansible_facts, data=None):
        """Populate the facts for interfaces

        :param connection: the device connection
        :param ansible_facts: Facts dictionary
        :param data: previously collected conf

        :rtype: dictionary
        :returns: facts
        """
        objs = []
        if not data:
            data = self.get_l2_interfaces_data(connection)

        facts = {"l2_interfaces": []}
        for interface in self.iter_interfaces(data):
            l2_interfaces_parser = L2_InterfacesTemplate(
                lines=interface.splitlines(), module=self._module)
            objs = l2_interfaces_parser.parse()

            if objs:
                self.format_vlans(objs)

            facts["l2_interfaces"].append(utils.remove_empties(objs))

        ansible_facts["ansible_network_resources"].pop("l2_interfaces", None)
        ansible_facts["ansible_network_resources"].update(facts)

        return ansible_facts

    def iter_interfaces(self, data):
        interfaces = re.compile(
            r"^interface port.+?(?=^!)", re.DOTALL | re.MULTILINE)
        return (i.group(0) for i in interfaces.finditer(data))

    def format_vlans(self, objs):
        if trunk := objs.get("trunk", None):
            if isinstance(trunk, dict):
                if vlans := trunk.get("vlans", None):
                    if isinstance(vlans, str):
                        trunk["vlans"] = vlans.split(",")
                    elif isinstance(vlans, int):
                        trunk["vlans"] = [str(vlans)]
                    else:
                        trunk["vlans"] = [str(vlan) for vlan in vlans]
