#
# -*- coding: utf-8 -*-
# Copyright 2019 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

"""
The aeos7 legacy fact class
It is in this file the configuration is collected from the device
for a given resource, parsed, and the facts tree is populated
based on the configuration.
"""

from __future__ import absolute_import, division, print_function


__metaclass__ = type


import platform
import re

from ansible.module_utils.six import iteritems
from ansible.module_utils.six.moves import zip

from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.aeos7 import (
    get_capabilities,
    run_commands,
)


class FactsBase(object):
    COMMANDS = list()

    def __init__(self, module):
        self.module = module
        self.facts = dict()
        self.warnings = list()
        self.responses = None

    def populate(self):
        self.responses = run_commands(
            self.module, commands=self.COMMANDS, check_rc=False)

    def run(self, cmd):
        return run_commands(self.module, commands=cmd, check_rc=False)


class Default(FactsBase):
    COMMANDS = ["show system"]

    def populate(self):
        super(Default, self).populate()
        self.facts.update(self.platform_facts())
        data = self.responses[0]
        if data:
            self.facts["serialnum"] = self.parse_serialnum(data)
            self.facts["model"] = self.parse_model(data)

    def parse_serialnum(self, data):
        match = re.search(r"Serial Number\s+: (\S+)", data)
        if match:
            return match.group(1)

    def parse_model(self, data):
        match = re.search(r"Hardware Model\s+: +(\S+)", data)
        if match:
            return match.group(1)

    def platform_facts(self):
        platform_facts = {}

        resp = get_capabilities(self.module)
        device_info = resp["device_info"]

        platform_facts["system"] = device_info["network_os"]

        for item in ("version",):
            val = device_info.get("network_os_{0}".format(item))
            if val:
                platform_facts[item] = val

        platform_facts["api"] = resp["network_api"]
        platform_facts["python_version"] = platform.python_version()

        return platform_facts


class Hardware(FactsBase):
    COMMANDS = ["show vmstat"]

    def populate(self):
        warnings = list()
        super(Hardware, self).populate()
        data = self.responses[0]

        if data:
            if vmstat_result := self.parse_vmstat(data):
                avm = int(vmstat_result[0])
                fre = int(vmstat_result[1])
                self.facts["memtotal_mb"] = (avm + fre) / 1024
                self.facts["memfree_mb"] = fre / 1024

    def parse_vmstat(self, data):
        for line in data.splitlines():
            m = re.search(r"\d+ \d+ \d+\s(\d+)\s+(\d+)", line)
            if m:
                return m.groups()


class Config(FactsBase):
    COMMANDS = ["show running-config"]

    def populate(self):
        super(Config, self).populate()
        data = self.responses[0]
        if data:
            data = re.sub(
                r"^Building configuration...\s+Current configuration : \d+ bytes\n",
                "",
                data,
                flags=re.MULTILINE,
            )
            self.facts["config"] = data


class Interfaces(FactsBase):
    COMMANDS = [
        "show interface status detail",
        "show ip interface brief",
    ]

    def populate(self):
        super(Interfaces, self).populate()

        self.facts["all_ipv4_addresses"] = list()

        data = self.responses[0]
        if data:
            self.facts["interfaces"] = self.parse_interfaces(data)

        data = self.responses[1]
        if data:
            self.facts["all_ipv4_addresses"] = self.populate_ip_address(data)

    def parse_ip_address(self, data):
        if m := re.search(r"^((loopback)||(manage)|(vlan \d+))+\s+(?P<address>\S+)", data):
            address = m.group("address")
            if address != "unassigned":
                return address

    def populate_ip_address(self, data):
        return [address for address in
                [self.parse_ip_address(line) for line in data.splitlines()]
                if address is not None]

    def parse_interface_detail_item(self, line, item_name):
        if m := re.match(rf"^{item_name}\s+:(.*)?$", line):
            return m.group(1).strip()

    def parse_interfaces(self, data):
        interface = {}
        interface_name = None
        for line in data.splitlines():
            if line == "":
                continue

            if port := self.parse_interface_detail_item(line, "Port"):
                interface_name = port
                interface[interface_name] = {}
                continue

            if interface_name:
                for item in (
                    ("Description", "description"),
                    ("Link status", "lineprotocol"),
                    ("Fix speed/duplex", "duplex"),
                    ("Media", "mediatype"),
                ):
                    parsed = self.parse_interface_detail_item(line, item[0])
                    if parsed is not None:
                        if parsed == "":
                            parsed = None
                        interface[interface_name][item[1]] = parsed
                        break

        return interface

    def populate_ipv4_interfaces(self, data):
        for key, value in data.items():
            self.facts["interfaces"][key]["ipv4"] = list()
            primary_address = addresses = []
            primary_address = re.findall(
                r"Internet address is (.+)$", value, re.M
            )
            addresses = re.findall(r"Secondary address (.+)$", value, re.M)
            if len(primary_address) == 0:
                continue
            addresses.append(primary_address[0])
            for address in addresses:
                addr, subnet = address.split("/")
                ipv4 = dict(address=addr.strip(), subnet=subnet.strip())
                self.add_ip_address(addr.strip(), "ipv4")
                self.facts["interfaces"][key]["ipv4"].append(ipv4)

    def add_ip_address(self, address, family):
        if family == "ipv4":
            self.facts["all_ipv4_addresses"].append(address)
