# -*- coding: utf-8 -*-
# Copyright 2022 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
"""
The aeos7 problem_info fact class
It is in this file the configuration is collected from the device
for a given resource, parsed, and the facts tree is populated
based on the configuration.
"""

from __future__ import absolute_import, division, print_function


__metaclass__ = type


import re

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common import utils
from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.rm_templates.problem_info import (
    HardwareProblemTemplate,
    SystemProblemTemplate,
)


class Problem_InfoFacts(object):
    """The aeos7 problem_info facts class"""

    def __init__(self, module, subspec="config", options="options"):
        self._module = module

    def get_hardware_data(self, connection):
        return connection.get("show hardware")

    def get_system_data(self, connection):
        return connection.get("show system")

    def populate_facts(self, connection, ansible_facts, data=None):
        """Populate the facts for problem_info network resource

        :param connection: the device connection
        :param ansible_facts: Facts dictionary
        :param data: previously collected conf

        :rtype: dictionary
        :returns: facts
        """
        facts = {}
        objs = []

        # parse native config using the Hardware Problem template
        data = self.get_hardware_data(connection)
        hardware_problem_parser = HardwareProblemTemplate(lines=data.splitlines(), module=self._module)
        objs = hardware_problem_parser.parse()

        # parse native config using the System Problem template
        data = self.get_system_data(connection)
        system_problem_parser = SystemProblemTemplate(lines=data.splitlines(), module=self._module)
        objs.update(system_problem_parser.parse())

        ansible_facts["ansible_network_resources"].pop("problem_info", None)
        facts = {"problem_info": objs}
        ansible_facts["ansible_network_resources"].update(facts)
        return ansible_facts
