# -*- coding: utf-8 -*-
# Copyright 2021 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function


__metaclass__ = type

"""
The aeos7 snmp_server fact class
It is in this file the configuration is collected from the device
for a given resource, parsed, and the facts tree is populated
based on the configuration.
"""

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common import utils

from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.rm_templates.snmp_server import (
    Snmp_serverTemplate,
)


class Snmp_serverFacts(object):
    """The aeos7 snmp_server facts class"""

    def __init__(self, module, subspec="config", options="options"):
        self._module = module

    def get_snmp_data(self, connection):
        return connection.get("show running-config | include ^snmp-server")

    def sort_list_dicts(self, objs):
        p_key = {
            "hosts": "ipaddr",
            "groups": "group",
            "communities": "name",
            "users": "username",
            "views": "view_name",
        }
        for k, _v in p_key.items():
            if k in objs:
                objs[k] = sorted(objs[k], key=lambda _k: str(_k[p_key[k]]))
        return objs

    def populate_facts(self, connection, ansible_facts, data=None):
        """Populate the facts for Snmp_server network resource

        :param connection: the device connection
        :param ansible_facts: Facts dictionary
        :param data: previously collected conf

        :rtype: dictionary
        :returns: facts
        """
        facts = {}
        objs = []
        params = {}

        if not data:
            data = self.get_snmp_data(connection)

        # parse native config using the Snmp_server template
        snmp_server_parser = Snmp_serverTemplate(
            lines=data.splitlines(), module=self._module)
        objs = snmp_server_parser.parse()

        if objs:
            self.sort_list_dicts(objs)
            if objs.get("enabled") is None:
                objs["enabled"] = True

        ansible_facts["ansible_network_resources"].pop("snmp_server", None)

        params = utils.remove_empties({"config": objs})

        facts["snmp_server"] = params.get("config", {})
        ansible_facts["ansible_network_resources"].update(facts)

        return ansible_facts
