# -*- coding: utf-8 -*-
# Copyright 2023 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function


__metaclass__ = type

"""
The aeos7_static_routes fact class
It is in this file the configuration is collected from the device
for a given resource, parsed, and the facts tree is populated
based on the configuration.
"""

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common import utils

from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.rm_templates.static_routes import (
    Static_RoutesTemplate,
)


class Static_RoutesFacts(object):
    """The aeos7_static_routes fact class"""

    def __init__(self, module, subspec="config", options="options"):
        self._module = module

    def get_static_routes_data(self, connection):
        return connection.get("show running-config | include ^ip route .+")

    def populate_facts(self, connection, ansible_facts, data=None):
        """Populate the facts for static_routes

        :param connection: the device connection
        :param ansible_facts: Facts dictionary
        :param data: previously collected conf

        :rtype: dictionary
        :returns: facts
        """

        if not data:
            data = self.get_static_routes_data(connection)

        static_routes_parser = Static_RoutesTemplate(
            lines=data.splitlines(), module=self._module)
        objs = static_routes_parser.parse()

        facts = {"static_routes": utils.remove_empties(objs)}
        ansible_facts["ansible_network_resources"].pop("static_routes", None)
        ansible_facts["ansible_network_resources"].update(facts)

        return ansible_facts
