#
# -*- coding: utf-8 -*-
# Copyright 2019 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
"""
The aeos7 vlans fact class
It is in this file the configuration is collected from the device
for a given resource, parsed, and the facts tree is populated
based on the configuration.
"""

from __future__ import absolute_import, division, print_function


__metaclass__ = type


import re

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common import utils

from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.rm_templates.vlans import (
    VlansTemplate,
)


class VlansFacts(object):
    """The aeos7 vlans fact class"""

    def __init__(self, module, subspec="config", options="options"):
        self._module = module

    def get_vlans_data(self, connection):
        return connection.get("show running-config")

    def populate_facts(self, connection, ansible_facts, data=None):
        """Populate the facts for vlans
        :param connection: the device connection
        :param ansible_facts: Facts dictionary
        :param data: previously collected conf
        :rtype: dictionary
        :returns: facts
        """

        facts = {"vlans": {}}
        if not data:
            data = self.get_vlans_data(connection)

        if vlan_database := re.search(r"^vlan database.+?^!", data,
                                      re.DOTALL | re.MULTILINE):
            vlans_parser = VlansTemplate(
                lines=vlan_database.group(0).splitlines(), module=self._module
            )
            objs = vlans_parser.parse()
            if vlans := objs.get("vlan"):
                objs["vlan"] = [val for val in vlans.values()]
                for vlan in objs["vlan"]:
                    if vlan.get("state") is None:
                        vlan["state"] = "enable"

            facts["vlans"].update(utils.remove_empties(objs))

        ansible_facts["ansible_network_resources"].pop("vlans", None)
        ansible_facts["ansible_network_resources"].update(facts)

        return ansible_facts
