# -*- coding: utf-8 -*-
# Copyright 2022 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function
from pickle import FALSE


__metaclass__ = type

"""
The Lag_interfaces parser templates file. This contains
a list of parser definitions and associated functions that
facilitates both facts gathering and native command generation for
the given network resource.
"""

import re

from copy import deepcopy

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common.rm_base.network_template import (
    NetworkTemplate,
)
from ansible_collections.ansible.netcommon.plugins.module_utils.network.common.utils import (
    dict_merge,
)


class Lag_InterfacesTemplate(NetworkTemplate):
    def __init__(self, lines=None, module=None):
        super(Lag_InterfacesTemplate, self).__init__(lines=lines, tmplt=self, module=module)

    # fmt: off
    PARSERS = [
        {
            "name": "lacp_system_priority",
            "getval": re.compile(
                r"""
                ^lacp\ssystem-priority
                (\s(?P<system_priority>\d+))
                $""", re.VERBOSE),
            "result": {
                "lacp_system_priority": "{{ system_priority }}"
            },
        },
        {
            "name": "lag_algorithm",
            "getval": re.compile(
                r"""
                ^link-aggregation\salgorithm
                (\s(?P<algorithm_base>\S+))
                (\s(?P<non_uc_dst_only>non-uc-dst-only))?
                $""", re.VERBOSE),
            "result": {
                "lag_load_balance": "{{ algorithm_base }}",
                "lag_non_uc_dst_only_enable": "{{ non_uc_dst_only }}" ,
            },
        },
        {
            "name": "port",
            "getval": re.compile(
                r"""
                ^interface\sport(\s+(?P<port>\S+))
                $""", re.VERBOSE,
            ),
            "result": {
                "port": "{{ port }}"
            },
        },
    ]

    PORT_PARSERS = [
        {
            "name": "member",
            "getval": re.compile(
                r"""
                ^\slink-aggregation
                (\s(?P<lagno>\d+))
                (\slacp\smode\s(?P<mode>active|passive))?
                (\skey\s(?P<keyno>\d+))?
                $""", re.VERBOSE
            ),
            'setval': "channel",
            "result": {
                "lag" : {
                    "lagno": "{{ lagno }}",
                    "lacp_mode": "{{ mode }}",
                    "key": "{{ keyno }}",
                },
            },
        },
        {
            "name": "lacp_port_priority",
            "getval": re.compile(
                r"""
                ^\slacp\sport-priority
                (\s(?P<priority>\d+))
                $""", re.VERBOSE),
            'setval': "channel",
            "result": {
                "lag" : {
                    "lacp_port_priority": "{{ priority }}"
                }
            },
        },
        {
            "name": "lacp_timeout",
            "getval": re.compile(
                r"""
                ^\slacp\stimeout
                (\s(?P<timeout>short|long))
                $""", re.VERBOSE),
            'setval': "channel",
            "result": {
                "lag" : {
                    "lacp_timeout": "{{ timeout }}"
                }
            },
        },
    ]
    # fmt: on

    def parse(self):
        def expand():
            nonlocal shared
            capdict = cap.groupdict()
            capdict = dict((k, v) for k, v in capdict.items() if v is not None)
            if parser.get("shared"):
                shared = capdict
            vals = dict_merge(capdict, shared)
            return self._deepformat(deepcopy(parser["result"]), vals)

        def additem(dst, src):
            for item_name in src:
                if "lagid" != item_name:
                    dst[item_name] = src[item_name]
        
        """parse"""
        result = {}
        shared = {}
        port_mode = False
        port_name = None
        ports_item = {}
        ports_member = []
        for line in self._lines:
            if '!' == line:
                if port_mode:
                    if len(ports_item) > 0:
                        ports_item["lag"].update(port_name)
                        ports_member.append(ports_item["lag"])
                        ports_item = {}
                    port_mode = False
                continue
            if port_mode:
                for parser in self.PORT_PARSERS:
                    cap = re.match(parser["getval"], line)
                    if cap:
                        res = expand()
                        ports_item = dict_merge(ports_item, res)
                        break
            else:
                for parser in self.PARSERS:
                    cap = re.match(parser["getval"], line)
                    if cap:
                        capdict = cap.groupdict()
                        capdict = dict((k, v) for k, v in capdict.items() if v is not None)
                        if parser.get("shared"):
                            shared = capdict
                        vals = dict_merge(capdict, shared)
                        res = self._deepformat(deepcopy(parser["result"]), vals)
                        if parser["name"] == "port":
                            port_name = res
                            port_mode = True
                        else:
                            result = dict_merge(result, res)
                        break
        uc = result.get("lag_non_uc_dst_only_enable")
        if uc:
            result["lag_non_uc_dst_only_enable"] = True
        if len(ports_member) > 0:
            result["member"] = ports_member
        if len(result) == 0:
            result = None
        return result
