# -*- coding: utf-8 -*-
# Copyright 2024 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function


__metaclass__ = type

"""
The Vlans parser templates file. This contains
a list of parser definitions and associated functions that
facilitates both facts gathering and native command generation for
the given network resource.
"""

import re

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common.rm_base.network_template import (
    NetworkTemplate,
)


class VlansTemplate(NetworkTemplate):
    def __init__(self, lines=None, module=None):
        super(VlansTemplate, self).__init__(
            lines=lines, tmplt=self, module=module
        )

    # fmt: off
    PARSERS = [
        {
            "name": "snap_nonzero_oui",
            "getval": re.compile(
                r"""
                ^\ssnap-nonzero-oui\senable
                $""", re.VERBOSE),
            "result": {
                "snap_nonzero_oui_enable": True
            },
        },
        {
            "name": "ignore_tag",
            "getval": re.compile(
                r"""
                ^\signore-tag\senable
                $""", re.VERBOSE),
            "result": {
                "ignore_tag_enable": True
            },
        },
        {
            "name": "tag_type",
            "getval": re.compile(
                r"""
                ^\stag-type
                (\s(?P<tag_type>\S+))
                $""", re.VERBOSE),
            "result": {
                "tag_type": "\"{{ tag_type }}\""
            },
        },
        {
            "name": "vlan.name",
            "getval": re.compile(
                r"""
                ^\svlan
                (\s(?P<vid>\S+))
                (\sname)
                (\s(?P<vlan_name>\S+))
                $""", re.VERBOSE),
            "result": {
                "vlan": {
                    "{{ vid }}": {
                        "vlan_id": "{{ vid }}",
                        "name": "{{ vlan_name }}"
                    },
                },
            },
        },
        {
            "name": "vlan.state",
            "getval": re.compile(
                r"""
                ^\svlan
                (\s(?P<vid>\S+))
                (\sstate)
                (\s(?P<disable>disable))
                $""", re.VERBOSE),
            "result": {
                "vlan": {
                    "{{ vid }}": {
                        "state": "disable"
                    },
                }
            },
        },
        {
            "name": "vlan.protocol",
            "getval": re.compile(
                r"""
                ^\svlan
                (\s(?P<vid>\S+))
                (\sadd\sprotocol)
                (\s(?P<protocol>\S+))
                (\s(?P<value>\S+))?
                $""", re.VERBOSE),
            "result": {
                "vlan": {
                    "{{ vid }}": {
                        "protocols": {
                            "{{ protocol }}": [
                                "\"{{ value if value is defined }}\"",
                            ],
                        },
                    },
                }
            },
        },
    ]
    # fmt: on
