#!/usr/bin/python
#
# This file is part of Ansible
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.
#
from __future__ import absolute_import, division, print_function


__metaclass__ = type

DOCUMENTATION = """
module: aeos7_command
author: 
short_description: Module to run commands on remote devices.
description:
- Sends arbitrary commands to an aeos7 node and returns the results read from the device.
- This module does not support running commands in configuration mode. Please use
  M(aeos7_config) to configure AEOS 7 devices.
version_added: 1.4.1
options:
  commands:
    description:
    - List of commands to send to the remote aeos7 device over the configured provider.
      The resulting output from the command is returned. If a command sent to the device requires
      answering a prompt, it is possible to pass a dict containing I(command), I(answer)
      and I(prompt). Common answers are 'y' or "\\r" (carriage return, must be double
      quotes). See examples.
    required: true
    type: list
    elements: raw
"""
EXAMPLES = r"""
- name: run show version on remote devices
  apresia.aeos7.aeos7_command:
    commands: show version

- name: run multiple commands on remote nodes
  apresia.aeos7.aeos7_command:
    commands:
    - show version
    - show interface status

- name: run commands that require answering a prompt
  apresia.aeos7.aeos7_command:
    commands:
      - command: copy memory-card config.txt flash-config
        prompt:
          - 'override?'
        answer:
          - "y"
"""
RETURN = """
stdout:
  description: The set of responses from the commands
  returned: always apart from low level errors (such as action plugin)
  type: list
  sample: ['...', '...']
stdout_lines:
  description: The value of stdout split into a list
  returned: always apart from low level errors (such as action plugin)
  type: list
  sample: [['...', '...'], ['...'], ['...']]
failed_conditions:
  description: The list of conditionals that have failed
  returned: failed
  type: list
  sample: ['...', '...']
"""
import time
import re

from ansible.module_utils._text import to_text
from ansible.module_utils.basic import AnsibleModule
from ansible_collections.ansible.netcommon.plugins.module_utils.network.common.parsing import (
    Conditional,
)
from ansible_collections.ansible.netcommon.plugins.module_utils.network.common.utils import (
    to_lines,
    transform_commands,
)

from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.aeos7 import (
    run_commands,
    aeos7_argument_spec,
)


def parse_commands(module, warnings):
    commands = transform_commands(module)
    if module.check_mode:
        for item in list(commands):
            if not item["command"].startswith("show"):
                warnings.append(
                    "Only show commands are supported when using check mode, not executing {0}".format(item["command"]))
                commands.remove(item)
    return commands


def main():
    """main entry point for module execution"""
    argument_spec = dict(
        commands=dict(type="list", elements="raw", required=True),
        wait_for=dict(type="list", elements="str", aliases=["waitfor"]),
        match=dict(default="all", choices=["all", "any"]),
        retries=dict(default=10, type="int"),
        interval=dict(default=1, type="int"),
    )
    argument_spec.update(aeos7_argument_spec)
    module = AnsibleModule(argument_spec=argument_spec,
                           supports_check_mode=True)
    warnings = list()
    result = {"changed": False, "warnings": warnings}
    commands = parse_commands(module, warnings)
    wait_for = module.params["wait_for"] or list()
    conditionals = []
    try:
        conditionals = [Conditional(c) for c in wait_for]
    except AttributeError as exc:
        module.fail_json(msg=to_text(exc))
    retries = module.params["retries"]
    interval = module.params["interval"]
    match = module.params["match"]
    while retries >= 0:
        if isinstance(commands, list) and \
            isinstance(commands[0], dict) and \
            re.match(r'show\s+tech', commands[0]['command']) is not None:
            responses = run_commands(module, commands, check_rc=False)
            if len(responses) == 1 and \
                'Invalid input detected at \'^\' marker.' in responses[0]:
                module.fail_json(msg=to_text(responses))
                return
        else:
            responses = run_commands(module, commands)
        for item in list(conditionals):
            if item(responses):
                if match == "any":
                    conditionals = list()
                    break
                conditionals.remove(item)
        if not conditionals:
            break
        time.sleep(interval)
        retries -= 1
    if conditionals:
        failed_conditions = [item.raw for item in conditionals]
        msg = "One or more conditional statements have not been satisfied"
        module.fail_json(msg=msg, failed_conditions=failed_conditions)
    result.update(
        {"stdout": responses, "stdout_lines": list(to_lines(responses))})
    module.exit_json(**result)


if __name__ == "__main__":
    main()
