#!/usr/bin/python
#
# This file is part of Ansible
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.
#
from __future__ import absolute_import, division, print_function


__metaclass__ = type

DOCUMENTATION = """
module: aeos7_config
author: 
short_description: Module to manage configuration sections.
description:
- Apresia AEOS 7 configurations use a simple block indent file syntax for segmenting configuration
  into sections.  This module provides an implementation for working with AEOS 7 configuration
  sections in a deterministic way.
version_added: 1.4.1
notes:
- To ensure idempotency and correct diff the configuration lines in the relevant module options should be similar to how they
  appear if present in the running configuration on device including the indentation.
options:
  lines:
    description:
    - The ordered set of commands that should be configured in the section. The commands
      must be the exact same commands as found in the device running-config to ensure
      idempotency and correct diff. Be sure to note the configuration command syntax as
      some commands are automatically modified by the device config parser.
    type: list
    elements: str
    aliases:
    - commands
  parents:
    description:
    - The ordered set of parents that uniquely identify the section or hierarchy the
      commands should be checked against.  If the parents argument is omitted, the
      commands are checked against the set of top level or global commands.
    type: list
    elements: str
  src:
    description:
    - Specifies the source path to the file that contains the configuration or configuration
      template to load.  The path to the source file can either be the full path on
      the Ansible control host or a relative path from the playbook or role root directory. This
      argument is mutually exclusive with I(lines), I(parents). The configuration lines in the
      source file should be similar to how it will appear if present in the running-configuration
      of the device including the indentation to ensure idempotency and correct diff.
    type: str
"""
EXAMPLES = """
- name: configure top level configuration
  apresia.aeos7.aeos7_config:
    lines: hostname {{ inventory_prompt }}

- name: configure interface settings
  apresia.aeos7.aeos7_config:
    lines:
    - description testinterface
    parents: interface port 1
"""
RETURN = """
updates:
  description: The set of commands that will be pushed to the remote device
  returned: always
  type: list
  sample: ['prompt foo', 'router ospf 1', 'router-id 192.0.2.1']
commands:
  description: The set of commands that will be pushed to the remote device
  returned: always
  type: list
  sample: ['prompt foo', 'router ospf 1', 'router-id 192.0.2.1']
"""
import json
import re

from ansible.module_utils._text import to_text
from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils.connection import ConnectionError
from ansible_collections.ansible.netcommon.plugins.module_utils.network.common.config import (
    NetworkConfig,
    dumps,
)

from ansible_collections.apresia.aeos7.plugins.module_utils.network.aeos7.aeos7 import (
    get_config,
    get_connection,
    get_defaults_flag,
    run_commands,
    aeos7_argument_spec,
)


def check_args(module, warnings):
    if module.params["multiline_delimiter"]:
        if len(module.params["multiline_delimiter"]) != 1:
            module.fail_json(
                msg="multiline_delimiter value can only be a single character")


def edit_config_or_macro(connection, commands):
    # only catch the macro configuration command,
    # not negated 'no' variation.
    if commands[0].startswith("macro name"):
        connection.edit_macro(candidate=commands)
    else:
        connection.edit_config(candidate=commands)


def get_candidate_config(module):
    candidate = ""
    if module.params["src"]:
        candidate = module.params["src"]
    elif module.params["lines"]:
        candidate_obj = NetworkConfig(indent=1)
        parents = module.params["parents"] or list()
        candidate_obj.add(module.params["lines"], parents=parents)
        candidate = dumps(candidate_obj, "raw")
    return candidate


def get_running_config(module, current_config=None, flags=None):
    running = module.params["running_config"]
    if not running:
        if not module.params["defaults"] and current_config:
            running = current_config
        else:
            running = get_config(module, flags=flags)
    return running


def save_config(module, result):
    result["changed"] = True
    if not module.check_mode:
        run_commands(module, "copy running-config flash-config")
    else:
        module.warn(
            "Skipping command `copy running-config flash-config` due to check_mode.  Configuration not copied to non-volatile storage"
        )


# exclude commad start
def replace_excluded_cmd_str(match):
    return '\n'


def exclude_commands(module, result, candidate):
    result["candidate_commands"] = candidate
    if module.params["exclude_commands"]:
        candidate_tmp = candidate
        exclude_commands = module.params["exclude_commands"]
        for exclude in exclude_commands:
            candidate_tmp = re.sub(
                exclude, replace_excluded_cmd_str, candidate_tmp)
        candidate = candidate_tmp

    result["include_commands"] = candidate
    return candidate
# exclude commad end


def main():
    """main entry point for module execution"""
    backup_spec = dict(filename=dict(), dir_path=dict(type="path"))
    argument_spec = dict(
        src=dict(type="str"),
        lines=dict(aliases=["commands"], type="list", elements="str"),
        parents=dict(type="list", elements="str"),
        before=dict(type="list", elements="str"),
        after=dict(type="list", elements="str"),
        match=dict(default="line", choices=["line", "strict", "exact", "none"]),
        replace=dict(default="line", choices=["line", "block"]),
        multiline_delimiter=dict(default="@"),
        running_config=dict(aliases=["config"]),
        defaults=dict(type="bool", default=False),
        backup=dict(type="bool", default=False),
        backup_options=dict(type="dict", options=backup_spec),
        save_when=dict(choices=["always", "never", "modified", "changed"], default="never"),
        diff_against=dict(choices=["flash", "running"]),
        diff_ignore_lines=dict(type="list", elements="str"),
        # exclude commad start
        exclude_commands=dict(type='list', elements="str"),
        # exclude commad end
    )
    argument_spec.update(aeos7_argument_spec)
    mutually_exclusive = [("lines", "src"), ("parents", "src")]
    module = AnsibleModule(
        argument_spec=argument_spec,
        mutually_exclusive=mutually_exclusive,
        supports_check_mode=True,)
    result = {"changed": False}
    warnings = list()
    check_args(module, warnings)
    result["warnings"] = warnings
    diff_ignore_lines = module.params["diff_ignore_lines"]
    config = None
    contents = None
    flags = get_defaults_flag(module) if module.params["defaults"] else []
    connection = get_connection(module)
    if module.params["backup"] or module._diff and module.params["diff_against"] == "running":
        contents = get_config(module, flags=flags)
        config = NetworkConfig(indent=1, contents=contents)
        if module.params["backup"]:
            result["__backup__"] = contents
    if any((module.params["lines"], module.params["src"])):
        match = module.params["match"]
        replace = module.params["replace"]
        path = module.params["parents"]
        candidate = get_candidate_config(module)
        # exclude commad start
        candidate = exclude_commands(module, result, candidate)
        # exclude commad end
        running = get_running_config(module, contents, flags=flags)
        try:
            response = connection.get_diff(
                candidate=candidate,
                running=running,
                diff_match=match,
                diff_ignore_lines=diff_ignore_lines,
                path=path,
                diff_replace=replace,
            )
        except ConnectionError as exc:
            module.fail_json(msg=to_text(exc, errors="surrogate_then_replace"))
        config_diff = response["config_diff"]
        if config_diff:
            commands = config_diff.split("\n")
            if module.params["before"]:
                commands[:0] = module.params["before"]
            if module.params["after"]:
                commands.extend(module.params["after"])
            result["commands"] = commands
            result["updates"] = commands

            # send the configuration commands to the device and merge
            # them with the current running config
            if not module.check_mode:
                if commands:
                    edit_config_or_macro(connection, commands)
            result["changed"] = True
    running_config = module.params["running_config"]
    flash_config = None
    if module.params["save_when"] == "always":
        save_config(module, result)
    elif module.params["save_when"] == "modified":
        output = run_commands(
            module, ["show running-config", "show flash-config"])
        running_config = NetworkConfig(
            indent=1, contents=output[0], ignore_lines=diff_ignore_lines)
        flash_config = NetworkConfig(
            indent=1, contents=output[1], ignore_lines=diff_ignore_lines)
        if running_config.sha1 != flash_config.sha1:
            save_config(module, result)
    elif module.params["save_when"] == "changed" and result["changed"]:
        save_config(module, result)
    if module._diff:
        if not running_config:
            output = run_commands(module, "show running-config")
            contents = output[0]
        else:
            contents = running_config

        # recreate the object in order to process diff_ignore_lines
        running_config = NetworkConfig(
            indent=1, contents=contents, ignore_lines=diff_ignore_lines)
        if module.params["diff_against"] == "running":
            if module.check_mode:
                module.warn(
                    "unable to perform diff against running-config due to check mode")
                contents = None
            else:
                contents = config.config_text
        elif module.params["diff_against"] == "flash":
            if not flash_config:
                output = run_commands(module, "show flash-config")
                contents = output[0]
            else:
                contents = flash_config.config_text
        if contents is not None:
            base_config = NetworkConfig(
                indent=1, contents=contents, ignore_lines=diff_ignore_lines)
            if running_config.sha1 != base_config.sha1:
                before, after = "", ""
                if module.params["diff_against"] in ("flash", "running"):
                    before = base_config
                    after = running_config
                result.update(
                    {"changed": True, "diff": {"before": str(before), "after": str(after)}})

    if result.get("changed") and any((module.params["src"], module.params["lines"])):
        msg = (
            "To ensure idempotency and correct diff the input configuration lines should be"
            " similar to how they appear if present in"
            " the running configuration on device"
        )
        if module.params["src"]:
            msg += " including the indentation"
        if "warnings" in result:
            result["warnings"].append(msg)
        else:
            result["warnings"] = msg

    module.exit_json(**result)


if __name__ == "__main__":
    main()
