#
# (c) 2016 Red Hat Inc.
#
# This file is part of Ansible
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.
#
from __future__ import absolute_import, division, print_function


__metaclass__ = type

import json
import re

from ansible.errors import AnsibleConnectionFailure
from ansible.module_utils._text import to_bytes, to_text
from ansible.utils.display import Display
from ansible_collections.ansible.netcommon.plugins.plugin_utils.terminal_base import TerminalBase


display = Display()


class TerminalModule(TerminalBase):
    ansi_re = [
        re.compile(rb'\x1b\[\?1h\x1b='),  # CSI ? 1 h ESC =
        re.compile(rb'\x08.'),            # [Backspace] .
        re.compile(rb"\x1b\[m"),          # ANSI reset code
        re.compile(rb'((?:\x9b|\x1b\x5b)[\x30-\x3f]*[\x20-\x2f]*[\x40-\x7e])')
    ]

    non_error_patterns = rb"|".join([
        rb"System 0x[0-9a-fA-F]+,([0-9a-fA-F]{2}-){5}[0-9a-fA-F]{2}",
        rb"Traffic statistics",
    ])

    terminal_stdout_re = [re.compile(
        rb"[\r\n]?[\w\+\-\.:\/\[\]]+(?:\([^\)]+\)){0,3}(?:[>#]) ?$")]

    privilege_level_re = re.compile(r"Current privilege level is (\d+)$")

    terminal_stderr_re = [
        re.compile(rb"% ?Error"),
        # re.compile(rb"^% \w+", re.M),
        re.compile(rb"% ?Bad secret"),
        re.compile(rb"[\r\n%] Bad passwords"),
        re.compile(rb"invalid input", re.I),
        re.compile(rb"(?:incomplete|ambiguous) command", re.I),
        re.compile(rb"connection timed out", re.I),
        re.compile(rb"[^\r\n]+ not found"),
        re.compile(rb"'[^']' +returned error code: ?\d+"),
        re.compile(rb"Bad mask", re.I),
        re.compile(rb"% ?(\S+) ?overlaps with ?(\S+)", re.I),
        re.compile(rb"[%\S] ?Error: ?[\s]+"),
        re.compile(rb"[%\S] ?ERROR: ?[\s]+"),
        re.compile(rb"\n ?ERROR:[ \S]"),
        re.compile(rb"[%\S] ?Informational: ?[\s]+", re.I),
        re.compile(rb"Command authorization failed"),
        re.compile(rb"Command Rejected: ?[\s]+", re.I),
        re.compile(rb"Verify Error:", re.I),
        re.compile(rb"VTY configuration is locked by other VTY"),
        re.compile(rb"\n% (?!" + non_error_patterns + rb").*"),
    ]

    terminal_config_prompt = re.compile(r"^.+\(config(-.*)?\)#$")

    general_prompts = (b'#', b'# ')

    become_enable = False

    def get_privilege_level(self):
        try:
            cmd = {"command": "show privilege"}
            result = self._exec_cli_command(
                to_bytes(json.dumps(cmd), errors="surrogate_or_strict"))
        except AnsibleConnectionFailure as e:
            raise AnsibleConnectionFailure(
                "unable to fetch privilege, with error: {0}".format(e.message))

        prompt = self.privilege_level_re.match(result)
        if not prompt:
            raise AnsibleConnectionFailure(
                "unable to check privilege level [{0}]".format(result))

        return int(prompt.group(1))

    def on_open_shell(self):
        if self.become_enable:
            try:
                cmds = ["configure terminal", "terminal length 0", "exit"]
                self._connection.run_commands(cmds, check_rc=True)
            except AnsibleConnectionFailure:
                raise AnsibleConnectionFailure("unable to configure terminal length 0")
        else:
            try:
                self._exec_cli_command(b"terminal length 0")
            except AnsibleConnectionFailure:
                raise AnsibleConnectionFailure("unable to set terminal parameters")

    def on_become(self, passwd=None):
        if self._get_prompt().endswith(self.general_prompts):
            return

        cmd = {"command": "enable"}
        if passwd:
            # Note: python-3.5 cannot combine u"" and r"" together.  Thus make
            # an r string and use to_text to ensure it's text on both py2 and py3.
            cmd["prompt"] = to_text(
                r"[\r\n]?(?:.*)?[Pp]assword: ?$", errors="surrogate_or_strict"
            )
            cmd["answer"] = passwd
            cmd["prompt_retry_check"] = True
        try:
            self._exec_cli_command(
                to_bytes(json.dumps(cmd), errors="surrogate_or_strict"))
            prompt = self._get_prompt()
            self.become_enable = True
        except AnsibleConnectionFailure as e:
            prompt = self._get_prompt()
            raise AnsibleConnectionFailure(
                "failed to elevate privilege to enable mode, at prompt [{0}] with error: {1}".format(prompt, e.message))

        if prompt is None or not prompt.endswith(self.general_prompts):
            raise AnsibleConnectionFailure(
                "failed to elevate privilege to enable mode, prompt [{0}]".format(prompt))

    def on_unbecome(self):
        prompt = self._get_prompt()
        if prompt is None:
            # if prompt is None most likely the terminal is hung up at a prompt
            return

        if b"(config" in prompt:
            self._exec_cli_command(b"end")
            self._exec_cli_command(b"disable")
        elif prompt.endswith(self.general_prompts):
            self._exec_cli_command(b"disable")
        self.become_enable = False
